function emMapOperation(em, prof, contour_type, powerflows, yax, normalized)
arguments
    em struct
    prof struct
    contour_type = "eff"
    powerflows string {mustBeText, mustBeMember(powerflows, ["all", "pe", "ps", "pt", "cd", "cs", "bc", "none", "bubbles"])}  = ["none"]
    yax string {mustBeMember(yax, ["power", "torque"])} = "torque"
    normalized = false
end
%emMapOperation
% Plot the e-machine operating points.
%
% Input arguments
% ---------------
% em : struct
%   E-machine data structure.
% prof : struct
%   Time profiles data structure.
% powerflows : string, optional
%   Specify one or more powerflows to represent in a string array. Specify:
%       "pe"  for pure electric (generator off);
%       "cd"  for charge-depleting;
%       "cs"  for charge-sustaining;
%       "bc"  for battery charging;
%       "all" for all powerflows.
%   The default is ["cd", "cs", "bc"].
%
% Outputs
% ---------------
% fig : Figure
%   Figure handle of the plot.
% ax : Axes
%   Axes handle of the plot.

%% Plot em Map
[fig, ax] = emMapPlot(em, contour_type, yax, normalized);

% Some compatibilty operations
if isfield(prof, "vehLoadFactor")
    pwrFlwCvName = "\lambda_{gen}";
    pwrFlwPrf = prof.vehLoadFactor;
elseif isfield(prof, "engTrqSplit")
    pwrFlwCvName = "\alpha_{eng}";
    pwrFlwPrf = prof.engTrqSplit;
else
    pwrFlwCvName = "missing";
    pwrFlwPrf = nan(size([prof.emSpd]));
end

if ~isfield(prof, "vehPwrFlw")
    vehPwrFlw = repmat("none", size([prof.emSpd]));
end

switch yax
    case "torque"
        yProf = [prof.emTrq];
    case "power"
        yProf = [prof.emSpd] .* [prof.emTrq] .* 1e-3; % kW;
end

%% Operating points scatter plot
if any(strcmp(powerflows, 'all'))
    powerflows = [ "pe", "ps", "pt", "cd", "cs", "bc"];
end

for n = 1:length(powerflows)
    powerflow = powerflows(n);
    idx = ismember(vehPwrFlw, powerflow);
    switch powerflow
        case 'pe'
            powerflow = "Pure electric";
            color = 'g';
        case 'ps'
            powerflow = "Power-split";
            color = 'b';
        case 'pt'
            powerflow = "Pure thermal";
            color = 'r';
        case 'none'
            powerflow = "Operating points";
            color = 'r';
        case 'cd'
            powerflow = "Charge depleting";
            color = 'b';
        case 'cs'
            powerflow = "Charge sustaining";
            color = 'r';
        case 'bc'
            powerflow = "Battery charging";
            color = 'k';
        case 'bubbles'
            color = "#1f77b4";
        otherwise
            continue
    end

    if strcmp(powerflow, "bubbles")
        idx = 1:length([prof.emSpd]);
        s = scatter([prof.emSpd] .* 30/pi, yProf, 100, 'o', ...
            'MarkerEdgeColor', "flat", 'MarkerEdgeAlpha', 0.1, ...
            'MarkerFaceColor', color, ...
            'MarkerFaceAlpha', 0.1, ...
            'DisplayName', "Operating points");
    else
        s = scatter([prof(idx).emSpd] .* 30/pi, yProf(idx), color, 'x', 'DisplayName', powerflow);
    end

    % Custom datatip
    s.DataTipTemplate.DataTipRows(1).Label = 'Speed';
    switch yax
        case "torque"
            s.DataTipTemplate.DataTipRows(2).Label = 'Torque';
        case "power"
            s.DataTipTemplate.DataTipRows(2).Label = 'Power';
    end
    s.DataTipTemplate.DataTipRows(end+1) = dataTipTextRow('Time, s', [prof(idx).time]);
    s.DataTipTemplate.DataTipRows(end+1) = dataTipTextRow('Gear Number', [prof(idx).gearNumber]);
    s.DataTipTemplate.DataTipRows(end+1) = dataTipTextRow(pwrFlwCvName, pwrFlwPrf(idx), '%.2f');
end

end
